<?php
/* Copyright (C) 2007-2008 	Laurent Destailleur	<eldy@users.sourceforge.net>
 * Copyright (C) 2010-2014 	Jean Heimburger 	<jean@tiaris.info>
 * Copyright (C) 2014-2016	Ferran Marcet	 	<fmarcet@2byte.es>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/**
 *      \file       importator/D_importator.class.php
 *      \ingroup    importator
 *      \brief      Import tool for dolibarr
 *        \version    $Id: ecom_log.class.php,v 1.1 2010-04-09 22:07:38 jean Exp $
 *        \author        jean Heimburger jean@tiaris.info
 *        \remarks    version 1.0
 */

// Put here all includes required by your class file
require_once DOL_DOCUMENT_ROOT . "/societe/class/societe.class.php";
require_once DOL_DOCUMENT_ROOT . "/product/class/product.class.php";
require_once DOL_DOCUMENT_ROOT . "/contact/class/contact.class.php";
require_once DOL_DOCUMENT_ROOT . "/comm/action/class/actioncomm.class.php";
require_once DOL_DOCUMENT_ROOT . "/user/class/user.class.php";
require_once DOL_DOCUMENT_ROOT . "/fourn/class/fournisseur.product.class.php";
require_once DOL_DOCUMENT_ROOT . "/core/lib/functions.lib.php";

/**
 *      \class      societe_import
 *      \brief      File import utility
 *        \remarks    importator module 2010-06-25
 */
class importator
{

    public $db;
    public $error;
    public $errors = array();

    public $process_resul;
    public $process_msg;
    public $filename;
    public $firstline;

    public $user;

    public function __construct($db, $filename, $user)
    {
        $this->db = $db;
        $this->error = '';
        $this->filename = $filename;
        $this->user = $user;
        $this->firstline = 0;
    }

// fonction d'imports de fichiers'
    public function validate_import($object)
    {
        global $langs, $conf;

        $this->process_msg = '';
        $error = 0;
        if (!utf8_check($this->filename)) $$this->filename = utf8_encode($this->filename);
        $fp = @fopen($this->filename, "r");
        if ($fp) {
            switch ($object) {
                case 'ImportStock':
                    $doliprod = new Product($this->db);

                    $i = 0;
                    while ($ligne = fgetcsv($fp, 1000, $conf->global->IMPORT_SEPARATOR)) {
                        $doliprod->id = '';
                        $i++;
                        if ($this->firstline && $i == 1) continue;

                        $ligne[0] = trim($ligne[0]);

                        if ($conf->global->IMPORT_BARCODE) {
                            $ligne[0] = $this->get_product_by_barcode($ligne[0]);
                        }

                        if ($doliprod->fetch('', $ligne[0]) <= 0) {
                            $this->process_msg .= $langs->trans("Untreated", $i) . " " . $langs->trans("ErrProdNoExist", $ligne[0], $doliprod->error) . "\n";
                            $error--;
                        }
                        $this->process_msg .= "\n";
                    }
                    break;
                case 'ImportProduct':
                    $doliprod = new Product($this->db);

                    $i = 0;
                    while ($ligne = fgetcsv($fp, 1000, $conf->global->IMPORT_SEPARATOR)) {
                        $doliprod->id = '';
                        $i++;
                        if ($this->firstline && $i == 1) continue;

                        if (!$ligne[0]) {
                            $this->process_msg .= $langs->trans("Untreated", $i) . " " . $langs->trans("ErrRefRequired") . "\n";
                            $error--;
                        }
                        $this->process_msg .= "\n";
                    }
                    break;
                case 'Importtarif':
                    $doliprod = new Product($this->db);
                    $societe = new Societe($this->db);
                    $i = 0;
                    while ($ligne = fgetcsv($fp, 1000, $conf->global->IMPORT_SEPARATOR)) {
                        $doliprod->id = '';
                        $i++;
                        if ($this->firstline && $i == 1) continue;

                        $ligne[0] = trim($ligne[0]);
                        $ligne[2] = trim($ligne[2]);

                        if ($doliprod->fetch('', $ligne[0]) <= 0) {
                            $this->process_msg .= $langs->trans("Untreated", $i) . " " . $langs->trans("ErrProdNoExist", $ligne[0], $doliprod->error) . "\n";
                            $error--;
                        }
                        if (!$ligne[1]) {
                            $this->process_msg .= $langs->trans("Untreated", $i) . " " . $langs->trans("ErrRefRequired") . "\n";
                            $error--;
                        }
                        if ($societe->fetch('', $ligne[2]) < 0) {
                            $this->process_msg .= $langs->trans("Untreated", $i) . " " . $langs->trans("ErrCompanyNoExist", $ligne[2], $societe->error) . "\n";
                            $error--;
                        }
                        $this->process_msg .= "\n";
                    }
                    break;
                case 'ImportThirtdparty':
                    $i = 0;
                    while ($ligne = fgetcsv($fp, 1000, $conf->global->IMPORT_SEPARATOR)) {
                        $i++;
                        if ($this->firstline && $i == 1) continue;
                        if (!$ligne[0]) {
                            $this->process_msg .= $langs->trans("Untreated", $i) . " " . $langs->trans("ErrNameRequired", $ligne[0]) . "\n";
                            $error--;
                        }
                    }
                    break;
                case 'ImportContact':
                    $i = 0;
                    $societe = new Societe($this->db);
                    while ($ligne = fgetcsv($fp, 1000, $conf->global->IMPORT_SEPARATOR)) {
                        $i++;
                        if ($this->firstline && $i == 1) continue;

                        // recherche id Société

                        if ($ligne[0])
                            $socid = $this->get_socbyclientcode($ligne[0]);
                        else if ($ligne[1])
                            $socid = $this->get_socbysuplliercode($ligne[1]);
                        if ($socid <= 0) {
                            $this->process_msg .= $langs->trans("Untreated", $i) . " " . $langs->trans("ErrCompanyRequired") . "\n";
                            $error--;
                        } else {
                            $societe->fetch($socid);
                            if (!$societe->id) {
                                $this->process_msg .= $langs->trans("Untreated", $i) . " " . $langs->trans("ErrCompanyNoExist", $ligne[0] . " " . $ligne[1]) . "\n";
                                $error--;
                            }
                        }
                        // test existence contact
                        if (empty($ligne[2])) {
                            $this->process_msg .= $langs->trans("Untreated", $i) . " " . $langs->trans("ErrNameRequired") . "\n";
                        }
                    }
                    break;
                /*case 'ImportActions':
                    $i=0;
                    $contact=new Contact($this->db);
                    $societe = new Societe($this->db);
                    $actioncomm = new ActionComm($this->db);
                    $actuser = new User($this->db);

                    while ($ligne = fgetcsv($fp,1000,";"))
                    {
                        $i++;
                        if ($this->firstline && $i== 1) continue;

                        if ($societe->fetch('',$ligne[0]) < 0 ){
                            $this->process_msg .= $langs->trans("Untreated",$i)." ".$langs->trans("ErrCompanyNoExist",$ligne[0])."\n";
                            $error--;
                        }
                        else $socid = $societe->id;

                        if (!$socid)
                        {
                            $this->process_msg .= $langs->trans("Untreated",$i)." ".$langs->trans("ErrCompanyNoExist",$ligne[0])."\n";
                            $error--;
                        }
                        $usertodo = '';
                        if ($ligne[9])
                        {
                            if ($actuser->fetch('',$ligne[9]) < 0) {
                                $this->process_msg .= $langs->trans("Untreated",$i)." ".$langs->trans("ErrUserNoExist",$ligne[9])."\n";
                                $error--;
                            }
                            else $usertodo = $actuser->id;
                        }
                        $userdone= '' ;
                        if ($ligne[10])
                        {
                            if ($actuser->fetch('',$ligne[10]) < 0) {
                                $this->process_msg .= $langs->trans("Untreated",$i)." ".$langs->trans("ErrUserNoExist",$ligne[10])."\n";
                            }
                            else $userdone = $actuser->id;
                        }
                        if ($ligne[6])
                        {
                            // voir date
                            $n = sscanf($ligne[6],"%02d/%02d/%04d", $d_day, $d_mon, $d_year);
                            if ($n==3) $datep=dol_mktime(12, 0, 0, $d_mon, $d_day, $d_year);
                            else {
                                $this->process_msg .= $langs->trans("Untreated",$i)." ".$langs->trans("ErrDateConversion",$ligne[6])."\n";
                                $error--;
                            }
                        }
                        if ($ligne[7])
                        {
                            // voir date
                            $n = sscanf($ligne[7],"%02d/%02d/%04d", $d_day, $d_mon, $d_year);
                            if ($n==3) $datep=dol_mktime(12, 0, 0, $d_mon, $d_day, $d_year);
                            else {
                                $this->process_msg .= $langs->trans("Untreated",$i)." ".$langs->trans("ErrDateConversion",$ligne[7])."\n";
                            }
                        }

                        $langs->trans("Treaty",$i)."\n";
                    }
                    break;	*/
            }
            fclose($fp);
        } else {
            $this->error = $langs->trans("ErrorOpenFile", $this->filename);
            $error = -1;
        }
        return $error;
    }

// création objet dans dolibarr $object = 'product' typeimport = création,  ou Modification ou Delete
    public function import2Dolibarr($object, $typeimport)
    {
        global $conf;
        global $langs;

        $this->process_msg = '';
        $error = 0;
        $fp = @fopen($this->filename, "r");
        if ($fp) {
            switch ($object) {
                case 'ImportStock':
                    $doliprod = new Product($this->db);
                    $i = 0;
                    while ($ligne = fgetcsv($fp, 1000, $conf->global->IMPORT_SEPARATOR)) {
                        $doliprod->id = '';
                        $i++;
                        if ($this->firstline && $i == 1) continue;

                        $ligne[0] = trim($ligne[0]);

                        if ($conf->global->IMPORT_BARCODE) {
                            $ligne[0] = $this->get_product_by_barcode($ligne[0]);
                        }

                        if ($doliprod->fetch('', $ligne[0]) <= 0) {
                            $this->process_msg .= $langs->trans("ErrProdNoExist", $ligne[0], $doliprod->error) . "\n";
                        } else {
                            $pid = $doliprod->id;
                            $doliprod->ref = $ligne[0];
                            $entrepot = $ligne[1];
                            $newstock = $ligne[2];
                            $price = price2num($ligne[3]);
                            if ($conf->global->IMPORT_TOTAL_STOCK) {
                                $doliprod->load_stock();
                                dol_syslog("stock produit " . $doliprod->stock_warehouse[$entrepot]->real . " entrepot " . $entrepot . " " . $doliprod->stock_reel, LOG_DEBUG);
                                // correction de stock
                                $delta = 0;
                                if ($newstock > $doliprod->stock_warehouse[$entrepot]->real) {
                                    $delta = $newstock - $doliprod->stock_warehouse[$entrepot]->real;
                                    $sens = 0;
                                } elseif ($newstock < $doliprod->stock_warehouse[$entrepot]->real) {
                                    $delta = $doliprod->stock_warehouse[$entrepot]->real - $newstock;
                                    $sens = 1;
                                }

                                if ($delta) {
                                    $res = $doliprod->correct_stock($this->user, $entrepot, $delta, $sens, $langs->trans("StockCorrection"), $price);
                                    if ($res < 0) {
                                        $this->process_msg .= $langs->trans("ErrMovStock", $ligne[0], $doliprod->error) . "\n";
                                        $error++;
                                    }
                                }
                                dol_syslog("maj stock delta = " . $delta . " sens " . $sens, LOG_DEBUG);
                            } else {
                                $res = $doliprod->correct_stock($this->user, $entrepot, abs($newstock), ($newstock > 0 ? 0 : 1), $langs->trans("StockCorrection"), $price);
                                if ($res < 0) {
                                    $this->process_msg .= $langs->trans("ErrMovStock", $ligne[0], $doliprod->error) . "\n";
                                    $error++;
                                }
                            }
                        }
                    }
                    break;
                case 'ImportProduct':
                    //$doliprod = new Product($this->db);
                    $i = 0;
                    while ($ligne = fgetcsv($fp, 1000, $conf->global->IMPORT_SEPARATOR)) {
                        set_time_limit(1);
                        $i++;
                        $doliprod = new Product($this->db);
                        $doliprod->id = '';
                        if ($this->firstline && $i == 1) continue;

                        $ligne[0] = trim($ligne[0]);
                        $ligne[1] = trim($ligne[1]);
                        $ligne[8] = trim($ligne[8]);
                        $ligne[9] = trim($ligne[9]);

                        if ($doliprod->fetch('', $ligne[0]) < 0) $this->process_msg .= $langs->trans("ErrProdNoExist", $ligne[0], $doliprod->error) . "\n";
                        else {
                            $pid = $doliprod->id;
                            $doliprod->ref = $ligne[0];
                            /*if($typeimport=='C'){
                                $doliprod->label = $ligne[1];
                                $doliprod->status = $ligne[2];
                                $doliprod->status_buy = 1;
                                $doliprod->description = $ligne[3];
                                $doliprod->price= $ligne[4];
                                $doliprod->tva_tx= $ligne[5];
                                $doliprod->weight=$ligne[6];
                                $doliprod->volume=$ligne[7];
                                $doliprod->barcode = $ligne[9];
                                $doliprod->type = $ligne[10];	 // toujours produit
                            }
                            else*/
                            {
                                if (version_compare(DOL_VERSION, 3.8) >= 0) {

                                    if (!empty($ligne[1])) $doliprod->label = $ligne[1];
                                    if (!empty($ligne[10])) $doliprod->barcode_type_code = $ligne[10];
                                }
                                if (!empty($ligne[1])) $doliprod->libelle = $ligne[1];//deprecated

                                if (!empty($ligne[2])) $doliprod->status = $ligne[2];
                                $doliprod->status_buy = 1;
                                if (!empty($ligne[3])) $doliprod->description = $ligne[3];
                                if (!empty($ligne[4])) $doliprod->price = $ligne[4];
                                if (!empty($ligne[5])) $doliprod->tva_tx = $ligne[5];
                                if (!empty($ligne[6])) $doliprod->weight = $ligne[6];
                                if (!empty($ligne[7])) $doliprod->volume = $ligne[7];
                                if (!empty($ligne[9])) $doliprod->barcode = $ligne[9];
                                if (!empty($ligne[10])) $doliprod->barcode_type = dol_getIdFromCode($this->db, $ligne[10], 'c_barcode_type', 'libelle', 'rowid');


                                if (!empty($ligne[11])) $doliprod->type = $ligne[11];
                            }
                            $doliprod->price_base_type = 'HT';

                            $this->db->begin;
                            switch ($typeimport) {
                                case 'C':
                                    if ($pid > 0) {
                                        if ($doliprod->update($pid, $this->user) < 0) {
                                            $this->process_msg .= $langs->trans("ErrProductUpdate", $ligne[0], $doliprod->error) . "\n";
                                            $error++;
                                        }

                                        if ($doliprod->updatePrice($doliprod->price, $doliprod->price_base_type, $this->user, '', '', $ligne[12]) < 0) {
                                            $this->process_msg .= $langs->trans("ErrProductUpdate", $ligne[0], $doliprod->error) . "\n";
                                            $error++;
                                        }

                                    } //$this->process_msg .= $langs->trans("Untreated",$i)." ".$langs->trans("ErrProdExist",$ligne[0])."\n";

                                    else {
                                        $sid = $doliprod->create($this->user);
                                        if ($sid < 0) {
                                            $this->process_msg .= $langs->trans("ErrProductCreate", $ligne[0], $doliprod->error) . "\n";
                                            $error++;
                                        } else {
                                            // image et code barre
                                            if ($ligne[8]) {
                                                $this->add_photo_web($conf->produit->dir_output, $ligne[8], $doliprod->id, $doliprod->ref);
                                            }
                                            /*if ($ligne[9]) {
                                                if ($doliprod->setValueFrom('fk_barcode_type', 2) < 0){
                                                    $this->process_msg .= $langs->trans("ErrProductCreate", $ligne[0], $doliprod->error)."\n"; // TODO paramétrer
                                                    $error++;
                                                }
                                                if ($doliprod->setValueFrom('barcode', $ligne[9]) < 0 ){
                                                    $this->process_msg .= $langs->trans("ErrProductCreate", $ligne[0], $doliprod->error)."\n";
                                                    $error++;
                                                }
                                            }*/

                                            $sql = "INSERT INTO " . MAIN_DB_PREFIX . "product_extrafields (fk_object) values (" . $sid . ")";
                                            $this->db->query($sql);

                                        }
                                    }
                                    break;
                                /*case 'M':
                                    if ($pid>0)
                                    {
                                        if ($doliprod->update($pid, $this->user) < 0){
                                            $this->process_msg .= $langs->trans("ErrProductUpdate", $ligne[0], $doliprod->error)."\n";
                                            $error++;
                                        }
                                        if (version_compare(DOL_VERSION, 3.5) >= 0){
                                            if ($doliprod->updatePrice($doliprod->price, $doliprod->price_base_type, $this->user) < 0){
                                                $this->process_msg .= $langs->trans("ErrProductUpdate", $ligne[0], $doliprod->error)."\n";
                                                $error++;
                                            }
                                        }
                                        else{
                                            if ($doliprod->updatePrice($doliprod->id, $doliprod->price, $doliprod->price_base_type, $this->user) < 0){
                                                $this->process_msg .= $langs->trans("ErrProductUpdate", $ligne[0], $doliprod->error)."\n";
                                                $error++;
                                            }
                                        }
                                    }
                                    else $this->process_msg .= $langs->trans("Untreated", $i).' '.$langs->trans("ErrProdNoExist", $ligne[0])."\n";
                                    break;*/
                                case 'D':
                                    if ($pid > 0) {
                                        if ($doliprod->delete($pid) < 0) {
                                            $this->process_msg .= $langs->trans("ErrProductDelete", $ligne[0], $doliprod->error) . "\n";
                                            $error++;
                                        }
                                    } else $this->process_msg .= $langs->trans("Untreated", $i) . ' ' . $langs->trans("ErrProdNoExist", $ligne[0]) . "\n";
                            }
                            if (!$error) {
                                $this->db->commit();
                            } else {
                                $this->db->rollback();
                            }
                        }
                    } // while
                    break;

                case 'ImportThirtdparty':

                    $i = 0;
                    //$societe = new Societe($this->db);

                    while ($ligne = fgetcsv($fp, 1000, $conf->global->IMPORT_SEPARATOR)) {
                        set_time_limit(1);
                        $i++;
                        $societe = new Societe($this->db);
                        $sid = '';
                        if ($this->firstline && $i == 1) continue;
                        if (!$ligne[0]) {
                            $this->process_msg .= $langs->trans("Untreated", $i) . " " . $langs->trans("ErrNameRequired") . "\n";
                            continue;
                        }

                        $ligne[0] = trim($ligne[0]);

                        // vérifier par code_client
                        if ($ligne[13])
                            $sid = $this->get_socbyclientcode($ligne[13]);
                        else if ($ligne[14])
                            $sid = $this->get_socbysuplliercode($ligne[14]);
                        if ($sid > 0) $societe->fetch($sid);
                        elseif ($societe->fetch('', $ligne[0]) > 0) {
                            $sid = $societe->id;
                        }

                        if ($ligne[12]) $pid = dol_getIdFromCode($this->db, $ligne[12], "c_country", "code", "rowid");
                        else $pid = '';
                        if ($pid <= 0) $pid = '';
                        $did = '';
                        $societe->id = $sid;
                        $societe->name = $ligne[0];
                        $societe->particulier = 0; //Société
                        $societe->address = $ligne[1] . "\n" . $ligne[2] . "\n" . $ligne[3];
                        $societe->zip = $ligne[4];
                        $societe->town = $ligne[5];
                        $societe->state_id = $did;
                        if ($ligne[12]) $societe->country_code = $ligne[12];
                        $societe->country_id = $pid;
                        dol_syslog("codes $pid " . $ligne[12], LOG_DEBUG);
                        $societe->phone = $ligne[6];
                        $societe->fax = $ligne[7];
                        $societe->email = $ligne[8];
                        $societe->url = $ligne[9];
                        $societe->idprof1 = $ligne[10];
                        switch ($ligne[11]) {
                            case '0' :
                                $societe->fournisseur = 0;
                                $societe->client = $ligne[11];
                                break;
                            case '1' :
                                $societe->fournisseur = 0;
                                $societe->client = $ligne[11];
                                break;
                            case '2' :
                                $societe->fournisseur = 0;
                                $societe->client = $ligne[11];
                                break;
                            case '10' :
                                $societe->client = 0;
                                $societe->fournisseur = 1;
                                break;
                            default:
                                ;
                                break;
                        }

                        if ($ligne[13]) $societe->code_client = $ligne[13];
                        if ($ligne[14]) $societe->code_fournisseur = $ligne[14];


                        /*if ($ligne[15]) $societe->array_options["options_zone"]=$ligne[15];
                        if (!empty($ligne[16])) $societe->array_options["options_CA"]=$ligne[16];

                        if (!empty($ligne[17]))
                        {
                               if ($ligne[17] <= 5) $societe->effectif_id = 1;
                               elseif ($ligne[17] <= 10) $societe->effectif_id = 2;
                               elseif ($ligne[17] <= 50) $societe->effectif_id = 3;
                               elseif ($ligne[17] <= 100) $societe->effectif_id = 4;
                               elseif ($ligne[17] <= 500) $societe->effectif_id = 5;
                               else $societe->effectif_id = 7;
                        }
                        dol_syslog("effectif " . $lige[17].'  '.$societe->effectif_id." ".print_r($societe->array_options, true), LOG_DEBUG);*/
                        $this->db->begin;
                        switch ($typeimport) {
                            case 'C':
                                if ($sid > 0) {
                                    if ($societe->update($sid, $this->user) < 0) {
                                        $this->process_msg .= $langs->trans("ErrCompanyUpdate", $ligne[0], $societe->error) . "\n";
                                        $error++;
                                    }
                                } else {
                                    $sid = $societe->create($this->user);
                                    if ($sid < 0) {
                                        $this->process_msg .= $langs->trans("ErrCompanyCreate", $ligne[0], $societe->error) . "\n";
                                        $error++;
                                    } else {
                                        $sql = "INSERT INTO " . MAIN_DB_PREFIX . "societe_extrafields (fk_object) values (" . $sid . ")";
                                        $this->db->query($sql);

                                    }
                                }
                                break;
                            /*case 'M':
                                if ($sid>0)
                                {
                                    if ($societe->update($sid, $this->user) < 0){
                                        $this->process_msg .= $langs->trans("ErrCompanyUpdate", $ligne[0], $societe->error)."\n";
                                        $error++;
                                    }
                                }
                                else $this->process_msg .= $langs->trans("Untreated", $i).' '.$langs->trans("CompanyNoExist", $ligne[0])."\n";
                                break;*/
                            case 'D':
                                if ($sid > 0) {
                                    if ($societe->delete($sid) < 0) {
                                        $this->process_msg .= $langs->trans("ErrCompanyDelete", $ligne[0], $societe->error) . "\n";
                                        $error++;
                                    }
                                } else $this->process_msg .= $langs->trans("Untreated", $i) . ' ' . $langs->trans("CompanyNoExist", $ligne[0]) . "\n";
                        }
                        if (!$error) {
                            $this->db->commit();
                        } else {
                            $this->db->rollback();
                        }
                    }
                    break;

                case 'ImportContact':
                    $i = 0;
                    //$contact=new Contact($this->db);
                    //$societe = new Societe($this->db);

                    while ($ligne = fgetcsv($fp, 1000, $conf->global->IMPORT_SEPARATOR)) {
                        set_time_limit(1);
                        $i++;
                        $contact = new Contact($this->db);
                        $societe = new Societe($this->db);
                        if ($this->firstline && $i == 1) continue;

                        $ligne[2] = trim($ligne[2]);
                        $ligne[3] = trim($ligne[3]);
                        //if ($societe->fetch('',$ligne[0]) < 0 ) $this->process_msg .= "erreur lecture Société "."\n";
                        if ($ligne[0])
                            $socid = $this->get_socbyclientcode($ligne[0]);
                        else if ($ligne[1])
                            $socid = $this->get_socbysuplliercode($ligne[1]);
                        if ($socid < 0) $this->process_msg .= $i . " " . $langs->trans("ErrCompanyRequired") . "\n";
                        else $societe->fetch($socid);

                        if (!$societe->id) {
                            $this->process_msg .= $langs->trans("ErrCompanyNoExist", $ligne[0] . " " . $ligne[1]) . "\n";
                            continue;
                        }

                        if (empty($ligne[2])) {
                            $this->process_msg .= $langs->trans("Untreated", $i) . " " . $langs->trans("ErrNameRequired") . "\n";
                            continue;
                        }
                        $contactid = $this->get_contact_id($socid, $ligne[2], $ligne[3]);

                        $contact->id = $contactid;
                        $contact->civility_id = $ligne[4];
                        $contact->lastname = $ligne[2];
                        $contact->firstname = $ligne[3];
                        if ($ligne[5] || $ligne[6] || $ligne[7]) $contact->address = $ligne[5] . "\n" . $ligne[6] . "\n" . $ligne[7];
                        else $contact->address = $societe->address;
                        if ($ligne[8]) $contact->zip = $ligne[8];
                        else $contact->zip = $societe->zip;
                        if ($ligne[9]) $contact->town = $ligne[9];
                        else $contact->town = $societe->town;
                        if ($ligne[10]) {
                            $pid = dol_getIdFromCode($this->db, $ligne[10], "c_country", "code", "rowid");
                            if ($pid <= 0) $pid = '';
                            $contact->country_id = $pid;
                            $contact->country_code = $ligne[10];
                        } else {
                            $contact->country_id = $societe->country_id;
                            $contact->country_code = $societe->country_code;
                        }
                        $contact->socid = $socid;                    // fk_soc
                        $contact->status = 1;
                        $contact->email = $ligne[11];
                        $contact->phone_pro = $ligne[12];
                        $contact->fax = $ligne[13];
                        $contact->phone_mobile = $ligne[14];
                        $contact->priv = 0;

                        $this->db - begin;
                        switch ($typeimport) {
                            case 'C':
                                if ($contactid > 0) {
                                    if ($contact->update($contactid, $this->user) < 0) {
                                        $this->process_msg .= $langs->trans("ErrContactUpdate", $ligne[2], $contact->error) . "\n";
                                        $error++;
                                    }
                                } else {
                                    $sid = $contact->create($this->user);
                                    if ($sid < 0) {
                                        $this->process_msg .= $langs->trans("ErrContactCreate", $ligne[2], $contact->error) . "\n";
                                        $error++;
                                    } else {
                                        $sql = "INSERT INTO " . MAIN_DB_PREFIX . "socpeople_extrafields (fk_object) values (" . $sid . ")";
                                        $this->db->query($sql);

                                    }
                                }
                                break;
                            /*case 'M':
                                if ($contactid>0)
                                {
                                    if ($contact->update($contactid, $this->user) < 0){
                                        $this->process_msg .= $langs->trans("ErrContactUpdate", $ligne[2], $contact->error)."\n";
                                        $error++;
                                    }
                                }
                                else $this->process_msg .= $langs->trans("Untreated", $i).' '.$langs->trans("ContactNoExist", $ligne[2])."\n";
                                break;*/
                            case 'D':
                                if ($contactid > 0) {
                                    if ($contact->delete($contactid) < 0) {
                                        $this->process_msg .= $langs->trans("ErrContactDelete", $ligne[2], $contact->error) . "\n";
                                        $error++;
                                    }
                                } else $this->process_msg .= $langs->trans("Untreated", $i) . ' ' . $langs->trans("ContactNoExist", $ligne[2]) . "\n";
                        }
                        if (!$error) {
                            $this->db->commit();
                        } else {
                            $this->db->rollback();
                        }
                    }
                    break;

                /*case 'ImportActions':
                    $i=0;
                    $contact=new Contact($this->db);
                    $societe = new Societe($this->db);
                    $actioncomm = new ActionComm($this->db);
                    $actuser = new User($this->db);

                    while ($ligne = fgetcsv($fp,1000,";"))
                    {
                        $i++;
                        if ($this->firstline && $i== 1) continue;

                        //if ($societe->fetch('',$ligne[0]) < 0 ) $this->process_msg .= "erreur lecture Société "."\n";
                        //else $socid = $societe->id;
                        $socid = $this->get_socbyclientcode($ligne[0]);
                        if ($socid < 0 ) $this->process_msg .= $langs->trans("Untreated", $i).' '.$langs->trans("ErrCompanyRequired")."\n";
                        else $societe->fetch($socid);
                        $socid = $societe->id;

                        if (!$socid)
                        {
                            $this->process_msg .= $langs->trans("ErrCompanyNoExist", $ligne[0])."\n";
                            continue;
                        }

                        //action sur un contact de la soc
                        if ($ligne[1])
                        {
                            $contactid = $this->get_contact_id($socid, $ligne[1], $ligne[2]) ;

                            if ($contactid < 0) {
                                $this->process_msg .= $langs->trans("Untreated", $i).' '.$langs->trans("ContactNoExist", $ligne[1].' '. $ligne[2])."\n";
                                // réinitialiser ??
                                continue;
                            }
                            else $contact->fetch($contactid);

                        }


                        $usertodo = '';
                        if ($ligne[9])
                        {
                            $usertodo=new User($this->db);
                            if ( $usertodo->fetch('',$ligne[9]) < 0 ) $this->process_msg .= $langs->trans("ErrUserNoExist", $ligne[9])."\n";
                        }
                        $userdone= '' ;
                        if ($ligne[10])
                        {
                            $usertodo=new User($this->db);
                            if ( $usertodo->fetch('',$ligne[10]) < 0 ) $this->process_msg .= $langs->trans("ErrUserNoExist", $ligne[10])."\n";

                        }
                        $datep = '';
                        if ($ligne[6])
                        {
                            // voir date
                            $n = sscanf($ligne[6],"%02d/%02d/%04d", $d_day, $d_mon, $d_year);
                            if ($n==3) $datep=dol_mktime(12, 0, 0, $d_mon, $d_day, $d_year);
                            if (!$datep) $this->process_msg .= $langs->trans("ErrDateConversion", $ligne[6])."\n";
                        }
                        else $datep ='';
                        $datef='';
                        if ($ligne[7])
                        {
                            // voir la date
                            $n = sscanf($ligne[7],"%02d/%02d/%04d", $d_day, $d_mon, $d_year);
                            if ($n==3)$datef=dol_mktime(12, 0, 0, $d_mon, $d_day, $d_year);
                            if (!$datef) $this->process_msg .= $langs->trans("ErrDateConversion", $ligne[7])."\n";
                        }
                        else $datef ='';
                        //$datef='';
                        $actioncomm->societe = $societe;
                        if ($ligne[1]) $actioncomm->contact = $contact;
                        else $actioncomm->contact = '';
                        $actioncomm->type_code = $ligne[3];
                        $actioncomm->priority = $ligne[4];
                        $actioncomm->location = '' ;
                        $actioncomm->label = $ligne[5];
                        $actioncomm->datep = $datep;
                        $actioncomm->datef = $datef;
                        $actioncomm->percentage = $ligne[8];
                        $actioncomm->usertodo = $usertodo;
                        $actioncomm->userdone = $userdone;
                        $actioncomm->note =$ligne[11];

                        switch ($typeimport)
                        {
                        case 'C':
                            $this->db->begin();
                            if ($actioncomm->add($this->user) < 0)
                            {
                                $this->process_msg .= $langs->trans("ErrActionCreate", $ligne[5], $actioncomm->error)."\n";
                                $this->db->rollback();
                            }
                            else $this->db->commit();
                            break;
                        case 'M':
                            $this->db->begin();
                            if($actioncomm->update($user) < 0){
                                $this->process_msg .= $langs->trans("ErrActionUpdate", $ligne[5], $actioncomm->error)."\n";
                                $this->db->rollback();
                            }
                            else $this->db->commit();
                            break;
                        case 'D':
                            $this->db->begin();
                            if($actioncomm->delete() < 0){
                                $this->process_msg .= $langs->trans("ErrActionDelete", $ligne[5], $actioncomm->error)."\n";
                                $this->db->rollback();
                            }
                            else $this->db->commit();
                            break;
                        }
                    }

                    break;*/

                case 'Importtarif' : // ref four

                    $i = 0;
                    $this->process_msg = '';
                    $error = 0;
                    $doliprod = new Product($this->db);
                    /*$product = new ProductFournisseur($this->db);
                    $societe = new Societe($this->db);*/

                    while ($ligne = fgetcsv($fp, 1000, $conf->global->IMPORT_SEPARATOR)) {
                        $doliprod->id = '';
                        $i++;

                        $doliprod = new Product($this->db);
                        $product = new ProductFournisseur($this->db);
                        $societe = new Societe($this->db);
                        if ($this->firstline && $i == 1) continue;

                        $ligne[0] = trim($ligne[0]);
                        $ligne[2] = trim($ligne[2]);

                        // recherche du fournisseur
                        if ($societe->fetch('', $ligne[2]) > 0) $sid = $societe->id;
                        else {
                            $this->process_msg .= $langs->trans("Untreated", $i) . " " . $langs->trans("ErrComppanyNoExist", $ligne[2]) . "\n";
                            $sid = '';
                        }

                        if ($doliprod->fetch('', $ligne[0]) > 0) $pid = $doliprod->id;
                        else {
                            $this->process_msg .= $langs->trans("Untreated", $i) . " " . $langs->trans("ErrProductNoExist", $ligne[0], $doliprod->error) . "\n";
                            $pid = '';
                        }

                        if ($sid > 0 && $pid > 0) {
                            $result = $product->fetch($doliprod->id);
                            if ($result > 0) {
                                $this->db->begin();
                                switch ($typeimport) {
                                    case 'C':
                                        $ret = $product->add_fournisseur($this->user, $sid, $ligne[1], price2num($ligne[3]));

                                        if ($ret < 0 && $ret != -3) {
                                            $this->process_msg .= $langs->trans("Untreated", $i) . " " . $langs->trans("ErrCreatePrice", $product->error) . "\n";
                                            $error++;
                                        }
                                        $ret = $product->update_buyprice(price2num($ligne[3]), price2num($ligne[4]), $this->user, 'HT', $societe, '', $ligne[1], price2num($ligne[5]));
                                        if ($ret < 0 && $ret != -3) {
                                            $this->process_msg .= $langs->trans("Untreated", $i) . " " . $langs->trans("ErrCreatePrice", $product->error) . "\n";
                                            $error++;
                                        }
                                        break;
                                    /*case 'M':
                                        {
                                            // gestion du prix obligatoire
                                            $supplier=new Fournisseur($this->db);
                                            $result=$supplier->fetch($sid);

                                            $ret=$product->update_buyprice($ligne[3], $ligne[4] , $this->user, 'HT', $supplier, '', $ligne[1], $ligne[5] );
                                            if ($ret < 0)
                                            {
                                                $this->process_msg .= $langs->trans("Untreated", $i)." ".$langs->trans("Qty").$ligne[3]. ", ".$langs->trans("Price").$ligne[4]." ".$langs->trans("ErrUpdatePrice", $product->error)."\n";
                                                $error ++;
                                            }

                                        }
                                        break;*/
                                    case 'D':
                                        // suprresion de la ligne avec le même nb d'article et le même prix
                                        $sql = "SELECT pfp.rowid FROM " . MAIN_DB_PREFIX . "product_fournisseur_price as pfp";
                                        $sql .= " WHERE pfp.quantity = '" . $ligne[3] . "' AND pfp.ref_fourn = '" . $ligne[1];
                                        $sql .= "' AND pfp.fk_soc = '" . $sid . "' AND pfp.fk_product='" . $pid . "'";
                                        $sql .= " AND pfp.entity = " . $conf->entity;

                                        $resql = $this->db->query($sql);
                                        if ($resql) {
                                            $obj = $this->db->fetch_object($resql);
                                            if ($obj->rowid > 0) {
                                                $result = $product->remove_product_fournisseur_price($obj->rowid);
                                            } else {
                                                $this->process_msg .= $langs->trans("Untreated", $i) . ' ' . $langs->trans("ErrDeletePrice", $product->error) . "\n";
                                                $error++;
                                            }
                                        } else {
                                            $this->process_msg .= "Error SQL= " . $sql . "\n";
                                            $error++;
                                        }
                                        break;
                                }//switch
                            }

                            if (!$error) {
                                $this->db->commit();
                            } else {
                                $this->db->rollback();
                            }
                        } // fournisseur trouvé
                    }// traitement ligne
                    // while reffour

                    break;
            } // fin switch
            fclose($fp);
        } else {
            $this->error = $langs->trans("ErrOpenFile") . $this->filename;
            $error = -1;
        }

        return $error;
    }

// fonctions de vérifications
    public function get_contact_id($socid, $contact_name, $contact_firstname)
    {
        if ($contact_name) $contact_name = $this->db->escape(trim($contact_name));
        else return 0;

        if ($contact_firstname) $contact_firstname = $this->db->escape(trim($contact_firstname));

        $sql = "SELECT rowid FROM " . MAIN_DB_PREFIX . "socpeople ";
        $sql .= "WHERE fk_soc ='" . $socid . "' AND lastname='" . $contact_name . "' ";
        if ($contact_firstname) $sql .= " AND firstname='" . $contact_firstname . "'";
        $sql .= " AND entity IN (" . getEntity('societe', 1) . ")";

        dol_syslog("D_importator::get_contact_id sql=" . $sql, LOG_DEBUG);
        $resql = $this->db->query($sql);
        if ($resql) {
            if ($this->db->num_rows($resql)) {
                // on renvoie toujours le prenmier
                $obj = $this->db->fetch_object($resql);
                $id = $obj->rowid;
            } else $id = 0;
        } else    $id = -1;

        return $id;
    }

    public function get_socbyclientcode($code_client)
    {
        if (empty($code_client)) return 0;
        $sql = "SELECT rowid FROM " . MAIN_DB_PREFIX . "societe WHERE code_client='" . trim($code_client) . "'";
        $sql .= " AND entity IN (" . getEntity('societe', 1) . ")";

        dol_syslog("D_importator::get_socbyclientcode sql=" . $sql, LOG_DEBUG);
        $resql = $this->db->query($sql);
        if ($resql) {
            if ($this->db->num_rows($resql)) {
                $obj = $this->db->fetch_object($resql);
                $id = $obj->rowid;
            } else $id = 0;
        } else $id = -1;

        return $id;
    }

    public function get_socbysuplliercode($code_supplier)
    {
        if (empty($code_supplier)) return 0;
        $sql = "SELECT rowid FROM " . MAIN_DB_PREFIX . "societe WHERE code_fournisseur='" . trim($code_supplier) . "'";
        $sql .= " AND entity IN (" . getEntity('societe', 1) . ")";

        dol_syslog("D_importator::get_socbysuppliercode sql=" . $sql, LOG_DEBUG);
        $resql = $this->db->query($sql);
        if ($resql) {
            if ($this->db->num_rows($resql)) {
                $obj = $this->db->fetch_object($resql);
                $id = $obj->rowid;
            } else $id = 0;
        } else $id = -1;

        return $id;
    }

    public function get_product_by_barcode($barcode)
    {
        $sql = "SELECT ref FROM " . MAIN_DB_PREFIX . "product WHERE barcode = " . $barcode;
        $sql .= " AND entity IN (" . getEntity('product', 1) . ")";
        $resql = $this->db->query($sql);

        if ($resql) {
            if ($this->db->num_rows($resql)) {
                $obj = $this->db->fetch_object($resql);
                $ref = $obj->ref;
            } else $ref = -1;
        } else $ref = -1;

        return $ref;
    }


    /**
     *  Deplace fichier recupere sur internet (utilise pour interface avec OSC)
     *
     * @param  string $sdir Repertoire destination finale
     * @param  string $file url de l'image
     * @param  int $idprod Product id where store the image
     * @return    void
     */
    public function add_photo_web($sdir, $file, $idprod, $refprod)
    {
        global $conf, $maxwidthsmall, $maxheightsmall, $maxwidthmini, $maxheightmini, $quality;

        require_once DOL_DOCUMENT_ROOT . '/core/lib/images.lib.php';        // This define also $maxwidthsmall, $quality, ...
        $object = new Product($this->db);
        $object->fetch($idprod);

        if (!empty($conf->global->PRODUCT_USE_OLD_PATH_FOR_PHOTO)) $dir = $sdir . "/" . get_exdir($idprod, 2, 0, 0, $object, 'product') . $idprod . "/photos/";
        else $dir = $sdir . "/" . dol_sanitizeFileName($refprod) . "/";


        $dir_osencoded = dol_osencode($dir);
        if (!file_exists($dir_osencoded)) {
            dol_syslog("Product Create " . $dir);
            dol_mkdir($dir);
        }

        if (file_exists($dir_osencoded)) {
            $content = @file_get_contents($file);
            if ($content) {
                $nom = basename($file);
                $im = fopen(dol_osencode($dir . $nom), 'wb');
                fwrite($im, $content);
                fclose($im);
            }
        }

        if (file_exists(dol_osencode($dir . $nom))) {
            $imgThumbSmall = vignette($dir . $nom, $maxwidthsmall, $maxheightsmall, '_small', $quality);

            $imgThumbMini = vignette($dir . $nom, $maxwidthmini, $maxheightmini, '_mini', $quality);
        }
    }

}

